<?php

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

/**
 * Manages gateway operations
 * @property  Currency_model Currency_model
 */
class ProcessPayment
{

    const DEFAULT_CURRENCY = 'usd';

    /**
     * @var CI_Controller
     */
    private $ci;

    private $selected_gateway;

    /**
     * @var array Company gateway settings
     */
    private $company_gateway_settings;

    /**
     * @var array Customer
     */
    private $customer;

    /**
     * @var string Error message
     */
    private $error_message;

    /**
     * @var string
     */
    private $currency = self::DEFAULT_CURRENCY;

    /**
     *
     * @var string External Id, can only be one per gateway
     */
    private $customer_external_entity_id;

    function __construct($params = null)
    {   
        $this->ci =& get_instance();

        $this->ci->load->model('Payment_gateway_model');
        $this->ci->load->model('Customer_model');
        $this->ci->load->library('session');
        $this->ci->load->model("Card_model");
        $this->ci->load->library('encrypt');
        $this->ci->load->model('Booking_model');
        $this->ci->load->model('Company_model');
        
        $company_id = $this->ci->session->userdata('current_company_id');

        if (isset($params['company_id'])) {
            $company_id = $params['company_id'];
        }
        
        if($company_id == 0 || $company_id == null){
            $last_segment = $this->ci->uri->segment($this->ci->uri->total_segments());
            $company_id = $last_segment;
        }
        $gateway_settings = $this->ci->Payment_gateway_model->get_payment_gateway_settings(
            $company_id
        );
               // prx($gateway_settings);
        if($gateway_settings)
        {
            $this->setCompanyGatewaySettings($gateway_settings);
            $this->setSelectedGateway($this->company_gateway_settings['selected_payment_gateway']);
            $this->populateGatewaySettings();
            $this->setCurrency();       
        } 

        $gateway_meta_data = $this->getGatewayCredentials();

        if($gateway_meta_data){
            $this->merchant_id = isset($gateway_meta_data['payment_gateway']['merchant_id']) ? $gateway_meta_data['payment_gateway']['merchant_id'] : '' ;

            $this->api_passcode = isset($gateway_meta_data['payment_gateway']['api_passcode']) ? $gateway_meta_data['payment_gateway']['api_passcode'] : '' ;
            
            $this->environment = isset($gateway_meta_data['payment_gateway']['environment']) ? $gateway_meta_data['payment_gateway']['environment'] : '' ;
        }

        // for worldline
        $this->worldline_url = ($this->ci->config->item('app_environment') == "development") ? "https://api.na.bambora.com" : "https://api.na.bambora.com";


    }

     /**
     * @param array $company_gateway_settings
     */
    public function setCompanyGatewaySettings($company_gateway_settings)
    {
        $this->company_gateway_settings = $company_gateway_settings;
    }

     /**
     * @param string $selected_gateway
     */
    public function setSelectedGateway($selected_gateway)
    {
        $this->selected_gateway = $selected_gateway;
    }


    private function populateGatewaySettings()
    {
        switch ($this->selected_gateway) {
            case 'worldline':
                $gateway_meta_data = json_decode($this->company_gateway_settings['gateway_meta_data'], true);

                $this->merchant_id = isset($gateway_meta_data['payment_gateway']['merchant_id']) ? $gateway_meta_data['payment_gateway']['merchant_id'] : '' ;

                $this->api_passcode = isset($gateway_meta_data['payment_gateway']['api_passcode']) ? $gateway_meta_data['payment_gateway']['api_passcode'] : '' ;
                $this->environment = isset($gateway_meta_data['payment_gateway']['environment']) ? $gateway_meta_data['payment_gateway']['environment'] : '' ;
                break;
        }
    }

    private function setCurrency()
    {
        // itodo some gateway currency maybe unavailable
        $this->ci->load->model('Currency_model');
        $currency       = $this->ci->Currency_model->get_default_currency($this->company_gateway_settings['company_id']);
        $this->currency = strtolower($currency['currency_code']);
    }

    
     /**
     * @return string
     */
    public function getErrorMessage()
    {
        return $this->error_message;
    }


    /**
     * @param string $error_message
     */
    public function setErrorMessage($error_message)
    {
        $this->error_message = $error_message;
    }


      /**
     * @param bool $publicOnly
     * @return array
     */
    public function getSelectedGatewayCredentials($publicOnly = false)
    {
        $credentials = $this->getGatewayCredentials($this->selected_gateway, $publicOnly);

        return $credentials;
    }

    /**
     * @param null $filter
     * @param bool $publicOnly
     * @return array
     */
    public function getGatewayCredentials($filter = null, $publicOnly = false)
    {
        $credentials                                     = array();
        $credentials['selected_payment_gateway']         = $this->selected_gateway; // itodo legacy
        
        $meta_data = json_decode($this->company_gateway_settings['gateway_meta_data'], true);
        
        $credentials['payment_gateway'] = array(
            'merchant_id' => isset($meta_data["merchant_id"]) ? $meta_data["merchant_id"] : "",
            'api_passcode' => isset($meta_data["api_passcode"]) ? $meta_data["api_passcode"] : "",

            'environment' => isset($meta_data["environment"]) ? $meta_data["environment"] : ""
            
        );

        $result                                = $credentials;

        if ($filter) {
            $result                             = isset($result[$filter]) ? $result[$filter] : $result['payment_gateway'];
            $result['selected_payment_gateway'] = $this->selected_gateway; // itodo legacy
        }

        return $result;
    }

     /**
     * @param $booking_id
     * @param $amount
     * @return charge_id
     */
    public function createBookingCharge($booking_id, $amount, $customer_id = null, $worldline_pre_auth = false)
    {
        $charge_id = null;
         if ($this->isGatewayPaymentAvailableForBooking($booking_id, $customer_id)) {
            try {
                $booking     = $this->ci->Booking_model->get_booking($booking_id);
                $customer_id = $customer_id ? $customer_id : $booking['booking_customer_id'];
                
                $customer_info    = $this->ci->Card_model->get_customer_cards($customer_id);
                $customer = "";
                if(isset($customer_info) && $customer_info){
                    
                    foreach($customer_info as $customer_data){
                        if(($customer_data['is_primary']) && !$customer_data['is_card_deleted']){
                            $customer = $customer_data;
                        }
                    } 
                }
                
                $customer    = json_decode(json_encode($customer), 1);

                $worldline_data = json_decode($customer['customer_meta_data'], true);
                $worldline_token = $worldline_data['token'];
                // $worldline_profile_code = $worldline_data['profile_code'];

               if($worldline_token)
               {        // use tokenex for payments
                    $customer['booking_id'] = $booking_id;
                    $customer['cusomer_id'] = $customer_id;
                    $charge = $this->make_payment($amount, $this->currency, $customer, $worldline_token, $worldline_pre_auth);

                    $charge_id = null;
                    if(isset($charge['success']) && $charge['success'])
                    {
                        if(isset($charge['charge_id']) && $charge['charge_id'])
                            $charge_id = $charge['charge_id'];
                        else
                        {
                           return $charge['authorization'];
                        }
                    }
                    else
                    {
                        if(isset($charge['success']) && $charge['errors'] == 'Unknown error'){
                            $this->setErrorMessage('Please, fill customer billing address (use two-letter country code)');
                        }else{
                            $this->setErrorMessage($charge['errors']);
                        }
                        
                    }
                }
                
                              
            } catch (Exception $e) {
                $error = $e->getMessage();
                $this->setErrorMessage($error);
            }
        }

        return $charge_id;
    }


     /**
     * @param $payment_id
     * @param $amount
     * @param $payment_type
     * @param $booking_id 
     * @return array
     */
    public function refundBookingPayment($payment_id, $amount, $payment_type, $booking_id = null)
    {
        $result = array("success" => true, "refund_id" => true);
        $this->ci->load->model('Payment_model');
        $this->ci->load->model('Customer_model');
   
        $payment = $this->ci->Payment_model->get_payment($payment_id);
        
        try {
            if ($payment['payment_gateway_used'] and $payment['gateway_charge_id']) {

                $customer    = $this->ci->Customer_model->get_customer($payment['customer_id']);
                
                unset($customer['cc_number']);
                unset($customer['cc_expiry_month']);
                unset($customer['cc_expiry_year']);
                unset($customer['cc_tokenex_token']);
                unset($customer['cc_cvc_encrypted']);

                $card_data = $this->ci->Card_model->get_active_card($payment['customer_id'], $this->ci->company_id);

                if(isset($card_data) && $card_data){
                    $customer['cc_number'] = $card_data['cc_number'];
                    $customer['cc_expiry_month'] = $card_data['cc_expiry_month'];
                    $customer['cc_expiry_year'] = $card_data['cc_expiry_year'];
                    $customer['cc_tokenex_token'] = $card_data['cc_tokenex_token'];
                    $customer['cc_cvc_encrypted'] = $card_data['cc_cvc_encrypted'];
                }
                
                $customer  = json_decode(json_encode($customer), 1);
                if($payment_id)
                {
                    if ($payment_type == 'full') {
                        $amount = abs($payment['amount']); // in cents, only positive
                    }
                    $result = $this->refund_payment($amount, $payment['gateway_charge_id']);
                    $result['payment_id'] = $payment_id;
                }
                
            }
        } catch (Exception $e) {
            $result = array("success" => false, "message" => $e->getMessage());
        }

        return $result;
    }

    /**
     * @param $gateway_secret_key
     * @param $amount
     * @param $currency
     * @param $customer 
     * @return array
     */
    public function make_payment($amount, $currency, $customer, $worldline_token, $worldline_pre_auth = false){       

        $api_url = $this->worldline_url;
        $method = '/v1/payments';
        $method_type = 'POST';

        $data = array(
                // "amount" => (int) round($amount * 100), // cents
                "amount" => (int) round($amount), // cents
                "currency" => "CAD",      // MUST match sandbox merchant
                "payment_method" => "token",
                // "payment_type"   => "SALE",   // 👈 THIS forces capture
                "token" => [
                    "name" => trim($customer['customer_name']),
                    "code" => $worldline_token // c50-xxxx from Custom Checkout
                ],
                // "order_number" => (string) $customer['booking_id']
                "order_number" => (string) $customer['booking_id'] . '-' . time()
        );

        // 🔐 AUTH HEADER (MANDATORY)
        $merchant_id  = (string) $this->merchant_id;
        $api_passcode = (string) $this->api_passcode;

        $auth = base64_encode($merchant_id . ':' . $api_passcode);

        $headers = [
            'Authorization: Passcode ' . $auth,
            'Content-Type: application/json'
        ];

        $response = $this->call_api($api_url, $method, $data, $headers, $method_type);

        $response = json_decode($response, true);

        if(
            isset($response['approved']) && 
            $response['approved'] == 1 &&
            isset($response['type']) && 
            $response['type'] == 'PA'
        ){

            if($worldline_pre_auth){
                return array('success' => true, 'charge_id' => $response['id'], 'type' => 'pre_auth');
            }

            $resp = $this->capture_payment($response['id'], $response['amount'], true);

            $resp = json_decode($resp, true);

            if(
                isset($resp['approved']) && 
                $resp['approved'] == 1 &&
                isset($resp['type']) && 
                $resp['type'] == 'PAC'
            ){
                return array('success' => true, 'charge_id' => $resp['id']);
            } else if (isset($resp['errors']) && $resp['errors']) {
                return array('success' => false, 'errors' => $resp['errors']);
            }
        }
    }

    public function capture_payment($auth_payment_id, $amount_cents, $is_direct_capture = false)
    {
        if(!$is_direct_capture){
            $payment = $this->ci->Payment_model->get_payment($auth_payment_id);
            $auth_payment_id = $payment['gateway_charge_id'];
        }

        $api_url = $this->worldline_url;
        $method = '/v1/payments/' . $auth_payment_id . '/completions';
        $method_type = 'POST';

        $data = [
            'amount' => (int) $amount_cents
        ];

        // 🔐 AUTH HEADER (MANDATORY)
        $merchant_id  = (string) $this->merchant_id;
        $api_passcode = (string) $this->api_passcode;

        $auth = base64_encode($merchant_id . ':' . $api_passcode);

        $headers = [
            'Authorization: Passcode ' . $auth,
            'Content-Type: application/json'
        ];

        $response = $this->call_api($api_url, $method, $data, $headers, $method_type);

        if(!$is_direct_capture){
            $response = json_decode($response, true);
            return array('success' => true, 'charge_id' => $response['id']);
        }

        return $response;
    }
 

    // public function create_profile_with_card($card_token)
    // {
    //     $api_url = 'https://api.na.bambora.com';
    //     $method  = '/v1/profiles';

    //     $data = [
    //         "language" => "en",
    //         "card" => [
    //             "token" => $card_token
    //         ]
    //     ];

    //     $merchant_id = (string) $this->merchant_id;
    //     $passcode    = '4C0D68BfB71b4da5887aF83963aBbfec';//$this->payment_api_passcode;

    //     $auth = base64_encode($merchant_id . ':' . $passcode);

    //     $headers = [
    //         'Authorization: Passcode ' . $auth,
    //         'Content-Type: application/json'
    //     ];

    //     $response = $this->call_api($api_url, $method, $data, $headers, 'POST');
    //     return json_decode($response, true);
    // }

    
    // public function createWorldlineCardProfile($customer){

    //     $api_url = $this->worldline_url;
    //     $method = '/v1/profiles';
    //     $method_type = 'POST';

    //     $customer_code = $customer['customer_id'];
    //     $token = $customer['worldline_token'];

    //     $data = array(
    //         "customer" => array(
    //             "customer_code" => (string) $customer_code
    //         ),
    //         "card" => array(
    //             "token" => $token
    //         )
    //     );


    //     $merchant_id = (string)$this->merchant_id; // ensure string
    //     $api_passcode = $this->api_passcode; // long hex string

    //     $auth = base64_encode($merchant_id . ':' . $api_passcode);

    //     $headers = [
    //         'Authorization: Passcode ' . $auth,
    //         'Content-Type: application/json'
    //     ];

    //     $response = $this->call_api($api_url, $method, $data, $headers, $method_type);

    //     $response = json_decode($response, true);

    //     prx($response, 1);

    //     // Check if API returned success

    //     if (
    //         isset($response['code']) && 
    //         $response['code'] == 1 && 
    //         isset($response['customer_code']) && 
    //         $response['customer_code']
    //     ) {
            
    //         return array(
    //             'success' => true, 
    //             'profile_code' => $response['customer_code']
    //         );
    //     }
    // }

    // public function verifyWorldlineCard($token, $profile_code) {

    //     $api_url = $this->worldline_url;
    //     $method  = '/v1/payments';

    //     $data = [
    //         "amount" => 1,
    //         "currency" => "CAD",
    //         "payment_method" => "card",
    //         "complete" => true,

    //         // 🔑 USE TOKEN HERE
    //         "card" => [
    //             "token" => $token
    //         ],

    //         // 🔑 LINK PAYMENT TO PROFILE
    //         "profile" => [
    //             "customer_code" => $profile_code
    //         ],

    //         "order" => [
    //             "order_number" => "INIT-" . time()
    //         ]
    //     ];

    //     $headers = [
    //         'Authorization: Passcode ' . $this->api_passcode,
    //         'Content-Type: application/json'
    //     ];

    //     $response = $this->call_api(
    //         $api_url,
    //         $method,
    //         $data,
    //         $headers,
    //         'POST'
    //     );

    //     $response = json_decode($response, true);

    //     if (!isset($response['code']) || $response['code'] != 1) {
    //         return [
    //             'success' => false,
    //             'error'   => $response
    //         ];
    //     }

    //     return [
    //         'success' => true,
    //         'payment' => $response
    //     ];
    // }


    // public function getWorldlineCards($customer_code) {
    //     // $api_url = $this->worldline_url . '/v1/customers/' . $customer_code . '/cards';

    //     echo $api_url = $this->worldline_url;
    //     echo "<br/>".$method = '/v1/profiles/' . $customer_code . '/cards';
    //     $method_type = 'GET';

    //     $merchant_id = (string)$this->merchant_id; // ensure string
    //     $api_passcode = $this->api_passcode; // long hex string

    //     $auth = base64_encode($merchant_id . ':' . $api_passcode);

    //     $data = array();

    //     $headers = [
    //         'Authorization: Passcode ' . $auth,
    //         'Content-Type: application/json'
    //     ];

    //     $response = $this->call_api($api_url, $method, $data, $headers, $method_type);
    //     $response = json_decode($response, true);

    //     prx($response, 1);

    //     if (!empty($response['cards'])) {
    //         // Return first card as example
    //         return [
    //             'success' => true,
    //             'card_id' => $response['cards'][0]['card_id'],
    //             'cards' => $response['cards']
    //         ];
    //     }

    //     return [
    //         'success' => false,
    //         'message' => 'No cards found for customer'
    //     ];
    // }

    /**
     * @param $gateway_secret_key
     * @param $amount
     * @param $gateway_charge_id
     * @return array
     */
    public function refund_payment($amount, $gateway_charge_id){

        $api_url = $this->worldline_url;
        $method  = '/v1/payments/' . $gateway_charge_id . '/returns';
        $method_type = 'POST';

        $data = [
            'amount' => (int) round($amount) // cents
        ];

        // 🔐 AUTH HEADER (MANDATORY)
        $merchant_id  = (string) $this->merchant_id;
        $api_passcode = (string) $this->api_passcode;

        $auth = base64_encode($merchant_id . ':' . $api_passcode);

        $headers = [
            'Authorization: Passcode ' . $auth,
            'Content-Type: application/json'
        ];

        // prx($data, 1);
        // prx($headers, 1);

        $response = $this->call_api($api_url, $method, $data, $headers, $method_type);

        $response = json_decode($response, true);
// 
        // prx($response, 1);

        if(
            isset($response['approved']) && 
            $response['approved'] == 1 &&
            isset($response['type']) && 
            $response['type'] == 'R'
        ){
            return array('success' => true, 'refund_id' => $response['id']);
        } else if (isset($response['errors']) && $response['errors']) {
            return array('success' => false, 'errors' => $response['errors']);
        }
    }

    /**
     * Can Booking perform payment operations
     *
     * @param $booking_id
     * @return bool
     */
    public function isGatewayPaymentAvailableForBooking($booking_id, $customer_id = null, $worldline_token = null)
    {
        $result = false;

        $booking       = $this->ci->Booking_model->get_booking($booking_id);
        
        $customer_id = $customer_id ? $customer_id : $booking['booking_customer_id'];
        
        $customer      = $this->ci->Customer_model->get_customer($customer_id);
        
        unset($customer['cc_number']);
        unset($customer['cc_expiry_month']);
        unset($customer['cc_expiry_year']);
        unset($customer['cc_tokenex_token']);
        unset($customer['cc_cvc_encrypted']);
        
        $card_data = $this->ci->Card_model->get_active_card($customer_id, $this->ci->company_id);
            
        if(isset($card_data) && $card_data){
            $customer['cc_number'] = $card_data['cc_number'];
            $customer['cc_expiry_month'] = $card_data['cc_expiry_month'];
            $customer['cc_expiry_year'] = $card_data['cc_expiry_year'];
            $customer['cc_tokenex_token'] = $card_data['cc_tokenex_token'];
            $customer['cc_cvc_encrypted'] = $card_data['cc_cvc_encrypted'];
            $customer['customer_meta_data'] = $card_data['customer_meta_data'];
        }
        
        $worldline_data = json_decode($customer['customer_meta_data'], true);
        $worldline_token = $worldline_data['token'];

        $hasTokenexToken = $worldline_token;
        

        if (
            $this->areGatewayCredentialsFilled()
            and $customer
            and $hasTokenexToken
        ) {
            $result = true;
        }

        return $result;
    }


    /**
     * @param $sql query
     * @return array
     */
    private function query($sql)
    {
        return $this->ci->db->query($sql)->result_array();
    }


    /**
     * @param $payment_type
     * @param $company_id
     * @return array
     */
    public function getPaymentGatewayPaymentType($payment_type, $company_id = null)
    {
        $payment_type = 'Worldline';
        $settings   = $this->getCompanyGatewaySettings();
        $company_id = $company_id ?: $settings['company_id'];

        $row = $this->query("select * from payment_type WHERE payment_type = '$payment_type' and company_id = '$company_id'");

        if (empty($row)) {
            // if doesn't exist - create
            $this->createPaymentGatewayPaymentType($payment_type, $company_id);
            $result = $this->getPaymentGatewayPaymentType($payment_type, $company_id);
        } else {
            $result = reset($row);
        }

        return $result;
    }

    /**
     * @param $payment_type
     * @param $company_id
     * @return int id
     */
    public function createPaymentGatewayPaymentType($payment_type, $company_id)
    {
        $this->ci->db->insert(
            'payment_type',
            array(
                'payment_type' => $payment_type,
                'company_id'   => $company_id,
                'is_read_only' => '1'
            )
        );

        return $this->ci->db->insert_id();
    }
    

    /**
     * @return array
     */
    public function getCompanyGatewaySettings()
    {
        return $this->company_gateway_settings;
    }

    /**
     * Checks if gateway settings are filled
     *
     * @return bool
     */
    public function areGatewayCredentialsFilled()
    {
        $filled                       = true;
        $selected_gateway_credentials = $this->getSelectedGatewayCredentials();

        foreach ($selected_gateway_credentials as $credential) {
            if (empty($credential)) {
                $filled = false;
            }
        }

        return $filled;
    }

    public function getSelectedGateway()
    {
        return $this->selected_gateway;
    }

    // public function call_api($api_url, $method, $data, $headers, $method_type = 'POST'){

    //     $url = $api_url . $method;
    //     $curl = curl_init();
    //     curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);
            
    //     if($method_type == 'GET'){

    //     }elseif($method_type == "DELETE"){
    //         curl_setopt($curl, CURLOPT_CUSTOMREQUEST, "DELETE");
    //         curl_setopt($curl, CURLOPT_POSTFIELDS, json_encode($data));
    //     } else {
    //         curl_setopt($curl, CURLOPT_POST, 1);
    //         curl_setopt($curl, CURLOPT_POSTFIELDS, json_encode($data));
    //     }
               
    //     curl_setopt($curl, CURLOPT_URL, $url);
    //     curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
    //     curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, 0);
    //     curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, 0);
    //     $response = curl_exec($curl);
        
    //     curl_close($curl);
        
    //     return $response;
    // }

    public function call_api($api_url, $endpoint, $data = [], $headers = [], $method_type = 'POST')
    {
        $url = rtrim($api_url, '/') . '/' . ltrim($endpoint, '/');

        $curl = curl_init();

        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_SSL_VERIFYHOST => 2,
            CURLOPT_SSL_VERIFYPEER => true,
        ]);

        switch (strtoupper($method_type)) {

            case 'GET':
                curl_setopt($curl, CURLOPT_HTTPGET, true);
                break;

            case 'DELETE':
                curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'DELETE');
                curl_setopt($curl, CURLOPT_POSTFIELDS, json_encode($data));
                break;

            case 'POST':
            default:
                curl_setopt($curl, CURLOPT_POST, true);
                curl_setopt($curl, CURLOPT_POSTFIELDS, json_encode($data));
                break;
        }

        $response = curl_exec($curl);

        if ($response === false) {
            $error = curl_error($curl);
            curl_close($curl);
            return json_encode(['curl_error' => $error]);
        }

        curl_close($curl);
        return $response;
    }


}






?>